<?php
/**
 * User Is Added Level
 *
 * @package     AutomatorWP\Integrations\Restrict_Content\Filters\User_Is_Added_Level
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Restrict_Content_User_Is_Added_Level_Filter extends AutomatorWP_Integration_Filter {

    public $integration = 'restrict_content';
    public $filter = 'restrict_content_user_is_added_level';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_filter( $this->filter, array(
            'integration'       => $this->integration,
            'label'             => __( 'User is in level', 'automatorwp-pro' ),
            'select_option'     => __( 'User <strong>is in level</strong>', 'automatorwp-pro' ),
            /* translators: %1$s: Condition. */
            'edit_label'        => sprintf( __( '%1$s', 'automatorwp-pro' ), '{access_level}' ),
            /* translators: %1$s: Condition. */
            'log_label'         => sprintf( __( '%1$s', 'automatorwp-pro' ), '{access_level}'),
            'options'           => array(
                'access_level' => automatorwp_utilities_ajax_selector_option( array(
                    'field'             => 'access_level',
                    'option_default'    => __( 'Level', 'automatorwp-pro' ),
                    'name'              => __( 'Access Level:', 'automatorwp-pro' ),
                    'option_custom'         => true,
                    'option_custom_desc'    => __( 'Access Level ID', 'automatorwp-pro' ),
                    'action_cb'         => 'automatorwp_restrict_content_get_access_levels',
                    'options_cb'        => 'automatorwp_restrict_content_options_cb_access_level',
                    'placeholder'       => __( 'Select an access level', 'automatorwp-pro' ),
                    'default'           => ''
                ) ),
            ),
        ) );

    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_filter    True if user deserves filter, false otherwise
     * @param stdClass  $filter             The filter object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $filter_options     The filter's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */
    public function user_deserves_filter( $deserves_filter, $filter, $user_id, $event, $filter_options, $automation ) {

        global $wpdb;
        
        // Shorthand
        $access_level_id = absint( $filter_options['access_level'] );

        // Bail if empty level to assign
        if( empty( $access_level_id ) ) {
            return;
        }

        $access_level = automatorwp_restrict_content_get_access_level_title( $access_level_id );

        // Bail if level not exists
        if( ! $access_level ) {
            $this->result = __( 'Filter not passed. Level does not exist.', 'automatorwp-pro' );
            return false;
        }

        $user_is_in_level = automatorwp_restrict_content_is_user_on_access_level( $user_id, $access_level_id );

        // Don't deserve if user does not have the level
        if( ! $user_is_in_level ) {
            $this->result = sprintf( __( 'Filter not passed. User does not meet the condition is in level "%1$s".', 'automatorwp-pro' ),
                $access_level
            );
            return false;
        }

        $this->result = sprintf( __( 'Filter passed. User is in level "%1$s".', 'automatorwp-pro' ),
            $access_level
        );

        return $deserves_filter;

    }

}

new AutomatorWP_Restrict_Content_User_Is_Added_Level_Filter();